% automated_pnet_workflow_example
% Christopher Zahasky
% 04/01/2019, subsection edited for repository example 09/16/2019

clear all
close all


%% Input
% Add flexible path to data
rootpath = pwd;
addpath(rootpath)

%%%%% You must download PNflow for this script to run
% PN flow can be downloaded here: https://github.com/aliraeini/pnflow
% Put this in a folder that doesn't change. IF YOUR FILE PATH HAS SPACES
% THIS MAY CAUSE ERRORS IN CALLING WITH 'SYSTEM' MATLAB FUNCTION
path2pnexe = [rootpath, '\pnflow-master\bin'];

% Pnextract base files
pnextract_base_file = 'pnextract_base_file.mhd';
pnflow_base_file = 'pnflow_base_file.dat';

% contact angle (defualt is 0)
ca = 40;
% interfacial tension, very important, default is around 40
ift = 51;

% image dimensions
image_dim = [333 333 333];

% voxel size (in microns)
voxel_size = 6;

% number or realization (in this example there is only 1 run)
number_runs = 1;
% color pc output by realization number
cc = jet(number_runs);

%%%%%%%% END INPUT %%%%%%%


%% PNM parameterization and data extraction
for iteration_number = 1:number_runs
    iteration_number
    
    % raw image name without .raw extension (automated naming necessary if
    % calling from this loop)
    image_name = 'raw_dry_333_invert_real_1';
    
    % Optional realization rejection check. This is commented out for this
    % example demonstration. This was done with MIJ interface
%     % Check if realization is one that should be removed
%     [porosity, grain_true]= FIJI_segmentation_check_function(...
%     image_name, rootpath, image_dim, real_check);
% 
%     phi_raw_image(iteration_number) = porosity;
%     % If porosity is very high or low or if 'known' grain area (as defined
%     % in real_check)
%     if grain_true == 0 || porosity > porosity_bounds(2) || porosity < porosity_bounds(1)
%         Pc{iteration_number} = 0;
%         phi(iteration_number) = 0;
%         perm(iteration_number) = 0; % in millidarcy
%         continue % if it is then kick back to beginning of loop
%     end
%     


    % create new folder
    new_folder_name = 'real_1_example';
    mkdir(new_folder_name)
    % copy file of raw image to new folder
    cd([rootpath, '\', new_folder_name])
    status = copyfile([rootpath, '\', image_name, '.raw']);
    cd(rootpath)
    
    % Pnextract files
    new_pnextract_filename = ['bent_', new_folder_name];
    
    % Change medial surface settings (clipROutx, clipROutyz, midRFrac,
    % RMedSurfNoise, lenNf, vmvRadRelNf, nRSmoothing, RCorsf, RCors)
    nRSmoothing = 1.5; % This parameter has some control on Pc curve but
    % strongly controls pnflow time, if set too high (above ~4) it can
    % increase run time by over a factor of 10x
    RMedSurfNoise = 0.7;
    MSS = [0.05, 0.05, 0.7, RMedSurfNoise, 0.6, 1, nRSmoothing, 0.15, 1];
    
    % Write pnextract file
    [pnextract_header_file]= pnextract_write_function(image_name, pnextract_base_file, ...
    new_pnextract_filename, new_folder_name, MSS, image_dim, voxel_size);
    
    % Pnflow read and write
    [pnflow_header_file, outfolder] = pnflow_write_function(image_name, ...
        pnflow_base_file, new_pnextract_filename, new_folder_name, ca, ift);
    
    %% Run simulations
    % change current directory to new folder
    cd([rootpath, '\', new_folder_name])
    
    % Run pnextract (suppress output with ';' after command)
    tic
    [status,cmdout] = system([path2pnexe, '\pnextract ', pnextract_header_file])
    toc

    % Run pnflow
    tic
    [status,cmdout] = system([path2pnexe, '\pnflow ', pnflow_header_file])
    toc
    
    %% Extract output, remove excess output, save data
    [pc_data, porosity, k, rel_perm_data] =  pnflow_output_extraction_function(outfolder);
    % record data (optional)
    Pc{iteration_number} = pc_data;
    phi(iteration_number) = porosity;
    perm(iteration_number) = k; % in millidarcy
    RP{iteration_number} = rel_perm_data;
 
    
    % plot Pc data (optional)
    figure(1)
    plot(Pc{iteration_number}(:,1), Pc{iteration_number}(:,2)./1000, 'color', cc(iteration_number, :))
    
    % Delete unneccesary output files (optional)
    delete([image_name, '_link1.dat'], [image_name, '_link2.dat'], ...
        [image_name, '_node1.dat'], [image_name, '_node2.dat'], ...
        [image_name, '_VElems.mhd'], [image_name, '_VElems.raw'])
    rmdir([outfolder, '_res'], 's')
    delete([image_name, '.raw'])
    
    
end