function [sol] = fit_distribution_skew_normal(bin_edges, y)
% A function for fitting a Normal PDF to histrogram data

% Calculate bin centroid locations
x = zeros(1, length(bin_edges) - 1); % Bin centroid locations
for i = 2:length(bin_edges)
	x(i-1) = mean([bin_edges(i), bin_edges(i-1)]);
end

% Find parameters of skewed Normal distribution
x0 = [x(round(end/2)), 0.1*x(round(end/2)), 0]; % Inital point (location, scale, shape)
lb = [min(x), 0, -50]; % Lower bound (location, scale, shape)
ub = [max(x), max(x), 50]; % Upper bound (location, scale, shape)

% Solver: lsqcurvefit
problem = createOptimProblem('lsqcurvefit', 'objective', @skew_normal_pdf, 'xdata', x, 'ydata', y, 'lb', lb, 'ub', ub, 'x0', x0);
ms = MultiStart('Display', 'off', 'UseParallel', true);
rs = RandomStartPointSet('NumStartPoints', 2000);
sol = run(ms, problem, rs);


% Residual between Normal PDF and data (residual sum of squares)
function [RSS] = skew_normal_residual(params, x, y)

% Unpack parameter values	
e = params(1); % Location parameter
w = params(2); % Scale parameter
a = params(3); % Shape parameter

% Caluclate vector of skew-normal PDF values (f)
phi_pdf = (1/sqrt(2*pi))*exp(-((x-e).^2)/2/(w^2));
phi_cdf = 0.5*(1 + erf(a*(x-e)/w/sqrt(2)));
f = 2*phi_pdf.*phi_cdf/w;

% Calculate residual sum of squares between skewed-normal and marginal posterior
RSS = sum((f - y).^2); % Residual sum of squares

end


% Skewed normal PDF values
function [f] = skew_normal_pdf(params, x)

% Unpack parameter values	
e = params(1); % Location parameter
w = params(2); % Scale parameter
a = params(3); % Shape parameter

% Caluclate vector of skew-normal PDF values (f)
phi_pdf = (1/sqrt(2*pi))*exp(-((x-e).^2)/2/(w^2));
phi_cdf = 0.5*(1 + erf(a*(x-e)/w/sqrt(2)));
f = 2*phi_pdf.*phi_cdf/w;

end

end